/* FILE: merror.c                               (D. Tottingham  02/23/91)

This is a collection of C functions that manage the error messages for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

er_abort ()                     display an error message then quit

EXTERNAL FUNCTIONS CALLED:

dm_reset ()                     reset this module and release all storage
dt_stop_ADC ()                  stop continuous data acquisition
o_write_logfile ()              write string to log file stream
s_reset_screen ()               reset the monochrome screen to text
s_screen_on ()                  return screen flag

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <stdio.h>
#include <malloc.h>

#include "mconst.h"
#include "mdemux.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mlog.h"
#include "mscreen.h"


/************************************************************************
                                DEFINES

************************************************************************/
#define FARHEAP         1
#define NEARHEAP        2


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE ER_MESSAGE em[] =  {
{ DM_NO_STORAGE         , "Unable to allocate storage in DM_GET_NEW_BUFFER."   , TRUE , TRUE  },
{ DSP_INVALID_FILE      , "Cannot open .fft file."                             , FALSE, TRUE  },
{ DSP_NO_BOARD          , "Symmetric Research TMS32020/C25 board not found."   , FALSE, TRUE  },
{ DT_FREQTOOHIGH        , "Requested aggregate frequency is too high."         , FALSE, TRUE  },
{ DT_FREQTOOLOW         , "Requested aggregate frequency is too low."          , FALSE, TRUE  },
{ DT_INVALID_BUFFERSIZE , "Aggregate buffer size exceeds 64K bytes."           , FALSE, TRUE  },
{ DT_INVALID_CHANNEL    , "Invalid channel number."                            , FALSE, TRUE  },
{ DT_INVALID_CHANNELSIZE, "ChannelBlocksize must be a power of 2."             , FALSE, TRUE  },
{ DT_INVALID_GAIN       , "Invalid gain number."                               , FALSE, TRUE  },
{ DT_INVALID_MUX        , "Invalid multiplexer installed."                     , FALSE, TRUE  },
{ DT_INVALID_SCANCOUNT  , "Number of station definitions must be a power of 2.", FALSE, TRUE  },
{ DT_IO_NOTINPROG       , "A/D error -- i/o not in progress."                  , FALSE, TRUE  },
{ DT_NO_ATLDRV          , "ATLAB device driver not found."                     , FALSE, TRUE  },
{ DT_NO_BOARD           , "Illegal DT28xx board number.  Check installation."  , FALSE, TRUE  },
{ DT_NO_STATIONS        , "No station definitions found in input file."        , FALSE, TRUE  },
{ DT_NO_STORAGE         , "Unable to allocate buffer in DT_INITIALIZE_BUFFER." , TRUE , TRUE  },
{ DT_OVERFLOW           , "ATLAB buffer overflow."                             , FALSE, TRUE  },
{ DT_TIMEOUT            , "ATLAB user timeout exceeded on input."              , FALSE, TRUE  },
{ DT_UNAVAILABLE_XB     , "Extended memory buffer is not available."           , FALSE, TRUE  },
{ F_DISK_FULL           , "Disk is full."                                      , FALSE, TRUE  },
{ F_INVALID_DRIVE       , "Invalid drive specification in path name."          , FALSE, TRUE  },
{ F_NO_STORAGE          , "Unable to allocate storage in F_INITIALIZE_PARAMS." , TRUE , TRUE  },
{ HS_NO_STORAGE         , "Unable to allocate storage in HS_LOCATE_EVENT."     , TRUE , TRUE  },
{ L_NO_STORAGE          , "Unable to allocate storage in L_LOCATE_EVENT."      , TRUE , TRUE  },
{ O_DISK_FULL           , "Disk is full."                                      , FALSE, FALSE },
{ O_INVALID_DRIVE       , "Invalid drive specification in log path name."      , FALSE, FALSE },
{ O_INVALID_LOGFILE     , "Unable to open log file."                           , FALSE, FALSE },
{ PK_NO_STORAGE         , "Unable to allocate storage in PK_PICK_EVENT."       , TRUE , TRUE  },
{ Q_NO_STORAGE          , "Unable to allocate storage in Q_SAVE_LINK."         , TRUE , TRUE  },
{ S_NO_STORAGE          , "Unable to allocate storage in S_PLOT_BUFFER."       , TRUE , TRUE  },
{ SP_INVALID_SIZE       , "Illegal spectral size."                             , FALSE, TRUE  },
{ SP_NO_BOARD           , "Symmetric Research TMS32020/C25 board not found."   , FALSE, TRUE  },
{ SP_NO_STORAGE         , "Unable to allocate storage in SP_COMPUTE_SPECTRUM." , TRUE,  TRUE  },
{ ST_INVALID_FILE       , "Invalid station file."                              , FALSE, TRUE  },
{ ST_NO_STORAGE         , "Unable to allocate storage in ST_ADD_STATION."      , TRUE , TRUE  },
{ U_NO_STORAGE          , "Unable to allocate storage in U_BUILD_TIMELINE."    , TRUE , TRUE  },
{ T_INVALID_CHANNEL     , "Invalid channel requested."                         , FALSE, TRUE  },
{ T_INVALID_LTASIZE     , "LTA window size must be a power of two."            , FALSE, TRUE  },
{ T_INVALID_STASIZE     , "STA window size must be a power of two."            , FALSE, TRUE  },
{ T_NO_STORAGE          , "Unable to allocate storage in T_SET_TRIGGER."       , TRUE , TRUE  },
{ X_INVALID_INPUTFILE   , "Cannot open input file."                            , FALSE, TRUE  },
{ DM_POWRSTOR           , "Unexpected PowrSTOR error."                         , FALSE, TRUE  },
{ DM_POWRSTOR_1         , "PowrSTOR initialization is invalid."                , FALSE, TRUE  },
{ DM_POWRSTOR_2         , "PowrSTOR initialization data is bad."               , FALSE, TRUE  },
{ DM_POWRSTOR_3         , "PowrSTOR is not initialized."                       , FALSE, TRUE  },
{ DM_POWRSTOR_4         , "Not enough memory available to PowrSTOR."           , FALSE, TRUE  },
{ DM_POWRSTOR_5         , "PowrSTOR file error."                               , FALSE, TRUE  },
{ DM_POWRSTOR_6         , "PowrSTOR expanded memory error."                    , FALSE, TRUE  },
{ DM_POWRSTOR_7         , "PowrSTOR extended memory error."                    , FALSE, TRUE  },
{ DM_POWRSTOR_8         , "PowrSTOR address is invalid."                       , FALSE, TRUE  },
{ DM_POWRSTOR_9         , "No more memory available to PowrSTOR."              , FALSE, TRUE  },
{ DM_POWRSTOR_10        , "PowrSTOR heap is invalid."                          , FALSE, TRUE  },
{ DM_POWRSTOR_11        , "No more heaps available to PowrSTOR"                , FALSE, TRUE  },
{ UNEXPECTED_ERROR      , "Unexpected error."                                  , FALSE, TRUE  }};


/*=======================================================================*
 *                             dump_heap                                 *
 *=======================================================================*/
/* Dump the far heap.                                                    */

PRIVATE
void dump_heap (stream, selector)
FILE *stream;
unsigned int selector;
{
   struct _heapinfo hinfo;
   long freebytes, usedbytes;
   int heapstatus;

   freebytes = usedbytes = 0;
   hinfo._pentry = NULL;
   while ((heapstatus =
          ((selector == FARHEAP) ? _fheapwalk(&hinfo) : _nheapwalk(&hinfo))) ==
          _HEAPOK) {
      fprintf (stream, "%6s block at %p of size %4.4u\n",
             (hinfo._useflag == _USEDENTRY ? "USED" : "FREE"),
             hinfo._pentry, hinfo._size);
      if (hinfo._useflag == _USEDENTRY)
         usedbytes += hinfo._size;
      else freebytes += hinfo._size;
   }

   fprintf (stream, "Used bytes: %ld, free bytes: %ld\n", usedbytes, freebytes);

   switch (heapstatus) {
      case _HEAPEMPTY:
         fprintf (stream, "OK: Empty heap.\n");
         break;
      case _HEAPEND:
         fprintf (stream, "OK: End of heap.\n");
         break;
      case _HEAPBADPTR:
         fprintf (stream, "ERROR: Bad pointer to heap.\n");
         break;
      case _HEAPBADBEGIN:
         fprintf (stream, "ERROR: Bad start of heap.\n");
         break;
      case _HEAPBADNODE:
         fprintf (stream, "ERROR: Bad node in heap.\n");
         break;
      default:
         fprintf (stream, "ERROR: Unrecognized heapstatus code.\n");
         break;
   }
}

/*=======================================================================*
 *                               quit0                                   *
 *=======================================================================*/
/* Quit without resetting the screen.                                    */

PRIVATE
void quit0 ()
{
   dt_stop_ADC ();
   dm_reset ();
   exit ( 1);
}

/*=======================================================================*
 *                             er_abort                                  *
 *=======================================================================*/
/* Display an error message then quit.                                   */

PUBLIC
void er_abort (message_id)
unsigned int message_id;
{
   static char message[80];
   FILE *stream;
   int i;

   if (s_screen_on())
      s_reset_screen();

   for (i = 0; em[i].id != UNEXPECTED_ERROR && em[i].id != message_id; i++);
   sprintf (message, "ERROR %3d: %s\n", message_id, em[i].text);
   printf (message);
   if (em[i].to_logfile)
      o_write_logfile (message);
   if (em[i].memory_dump) {
      printf ("Heaps have been dumped in dumpheap.@@@\n");
      if (stream = fopen ("dumpheap.@@@", "w")) {
         fprintf (stream, "**** Dump of FAR heap ****\n");
         dump_heap (stream, FARHEAP);
         fprintf (stream, "\n\n**** Dump of NEAR heap ****\n");
         dump_heap (stream, NEARHEAP);
         fclose (stream);
      }
   }
   quit0();
}
